within Modelica_LinearSystems2.Math.Matrices.LAPACK;
function dhseqr
  "Compute eingenvalues of a matrix A using lapack routine DHSEQR for Hessenberg form matrix"
  input Real H[:,size(H, 1)];
  input Integer lwork=max(1, size(H, 1));
  input Boolean eigenValuesOnly=true;
  input String compz="N";
  input Real Z[:,:]=H;
  output Real alphaReal[size(H, 1)]
    "Real part of alpha (eigenvalue=(alphaReal+i*alphaImag))";
  output Real alphaImag[size(H, 1)]
    "Imaginary part of alpha (eigenvalue=(alphaReal+i*alphaImag))";
  output Integer info;
  output Real Ho[:,:]=H;
  output Real Zo[:,:]=Z;
  output Real work[max({lwork, size(H, 1),1})];

protected
  Integer n=size(H, 1);
  String job=if eigenValuesOnly then "E" else "S";
  Integer ilo=1;
  Integer ihi=n;
  Integer ldh=max(n, 1);
  Integer lw=if lwork == -1 then -1 else max(lwork, size(H, 1));

external "Fortran 77" dhseqr(
    job,
    compz,
    n,
    ilo,
    ihi,
    Ho,
    ldh,
    alphaReal,
    alphaImag,
    Zo,
    ldh,
    work,
    lw,
    info) annotation(Library = {"lapack"});

  annotation (Documentation(info="Lapack documentation:

   Purpose
   =======

   DHSEQR computes the eigenvalues of a real upper Hessenberg matrix H
   and, optionally, the matrices T and Z from the Schur decomposition
   H = Z T Z**T, where T is an upper quasi-triangular matrix (the Schur
   form), and Z is the orthogonal matrix of Schur vectors.

   Optionally Z may be postmultiplied into an input orthogonal matrix Q,
   so that this routine can give the Schur factorization of a matrix A
   which has been reduced to the Hessenberg form H by the orthogonal
   matrix Q:  A = Q*H*Q**T = (QZ)*T*(QZ)**T.

   Arguments
   =========

   JOB     (input) CHARACTER*1
           = 'E':  compute eigenvalues only;
           = 'S':  compute eigenvalues and the Schur form T.

   COMPZ   (input) CHARACTER*1
           = 'N':  no Schur vectors are computed;
           = 'I':  Z is initialized to the unit matrix and the matrix Z
                   of Schur vectors of H is returned;
           = 'V':  Z must contain an orthogonal matrix Q on entry, and
                   the product Q*Z is returned.

   N       (input) INTEGER
           The order of the matrix H.  N >= 0.

   ILO     (input) INTEGER
   IHI     (input) INTEGER
           It is assumed that H is already upper triangular in rows
           and columns 1:ILO-1 and IHI+1:N. ILO and IHI are normally
           set by a previous call to DGEBAL, and then passed to SGEHRD
           when the matrix output by DGEBAL is reduced to Hessenberg
           form. Otherwise ILO and IHI should be set to 1 and N
           respectively.
           1 <= ILO <= IHI <= N, if N > 0; ILO=1 and IHI=0, if N=0.

   H       (input/output) DOUBLE PRECISION array, dimension (LDH,N)
           On entry, the upper Hessenberg matrix H.
           On exit, if JOB = 'S', H contains the upper quasi-triangular
           matrix T from the Schur decomposition (the Schur form);
           2-by-2 diagonal blocks (corresponding to complex conjugate
           pairs of eigenvalues) are returned in standard form, with
           H(i,i) = H(i+1,i+1) and H(i+1,i)*H(i,i+1) < 0. If JOB = 'E',
           the contents of H are unspecified on exit.

   LDH     (input) INTEGER
           The leading dimension of the array H. LDH >= max(1,N).

   WR      (output) DOUBLE PRECISION array, dimension (N)
   WI      (output) DOUBLE PRECISION array, dimension (N)
           The real and imaginary parts, respectively, of the computed
           eigenvalues. If two eigenvalues are computed as a complex
           conjugate pair, they are stored in consecutive elements of
           WR and WI, say the i-th and (i+1)th, with WI(i) > 0 and
           WI(i+1) < 0. If JOB = 'S', the eigenvalues are stored in the
           same order as on the diagonal of the Schur form returned in
           H, with WR(i) = H(i,i) and, if H(i:i+1,i:i+1) is a 2-by-2
           diagonal block, WI(i) = sqrt(H(i+1,i)*H(i,i+1)) and
           WI(i+1) = -WI(i).

   Z       (input/output) DOUBLE PRECISION array, dimension (LDZ,N)
           If COMPZ = 'N': Z is not referenced.
           If COMPZ = 'I': on entry, Z need not be set, and on exit, Z
           contains the orthogonal matrix Z of the Schur vectors of H.
           If COMPZ = 'V': on entry Z must contain an N-by-N matrix Q,
           which is assumed to be equal to the unit matrix except for
           the submatrix Z(ILO:IHI,ILO:IHI); on exit Z contains Q*Z.
           Normally Q is the orthogonal matrix generated by DORGHR after
           the call to DGEHRD which formed the Hessenberg matrix H.

   LDZ     (input) INTEGER
           The leading dimension of the array Z.
           LDZ >= max(1,N) if COMPZ = 'I' or 'V'; LDZ >= 1 otherwise.

   WORK    (workspace/output) DOUBLE PRECISION array, dimension (LWORK)
           On exit, if INFO = 0, WORK(1) returns the optimal LWORK.

   LWORK   (input) INTEGER
           The dimension of the array WORK.  LWORK >= max(1,N).

           If LWORK = -1, then a workspace query is assumed; the routine
           only calculates the optimal size of the WORK array, returns
           this value as the first entry of the WORK array, and no error
           message related to LWORK is issued by XERBLA.

   INFO    (output) INTEGER
           = 0:  successful exit
           < 0:  if INFO = -i, the i-th argument had an illegal value
           > 0:  if INFO = i, DHSEQR failed to compute all of the
                 eigenvalues in a total of 30*(IHI-ILO+1) iterations;
                 elements 1:ilo-1 and i+1:n of WR and WI contain those
                 eigenvalues which have been successfully computed.

   =====================================================================  "));
end dhseqr;
